<?php

/**
 * JCH Optimize - Performs several front-end optimizations for fast downloads
 *
 * @package   jchoptimize/wordpress-platform
 * @author    Samuel Marshall <samuel@jch-optimize.net>
 * @copyright Copyright (c) 2020 Samuel Marshall / JCH Optimize
 * @license   GNU/GPLv3, or later. See LICENSE file
 *
 * If LICENSE file missing, see <http://www.gnu.org/licenses/>.
 */

namespace JchOptimize\Helper;

class TabSettings
{
	public static function getSettingsArray()
	{
		$aTabs          = get_class_methods( __CLASS__ );
		$aSettingsArray = [];

		foreach ( $aTabs as $tab )
		{
			if ( $tab == str_replace( __CLASS__ . '::', '', __METHOD__ ) )
			{
				continue;
			}

			$aSettingsArray = array_merge( $aSettingsArray, self::$tab() );
		}

		return $aSettingsArray;
	}

	public static function generalTab()
	{
		return [
			/**
			 * Miscellaneous
			 */
			'miscellaneousSection'    => [
				'pro_downloadid'          => [
					__( 'Download ID', 'jch-optimize' ),
					__( 'You\'ll find your Download ID in your account. Enter your Download ID here to enable automatic updates of the PRO version and to access our Optimize Image API.', 'jch-optimize' ),
					true
				],
				'debug'                   => [
					__( 'Debug Plugin', 'jch-optimize' ),
					__( 'This option will add the \'commented out\' url of the individual files inside the combined file above the contents that came from that file. This is useful when configuring the plugin and trying to resolve conflicts. <p>This will also add a Profiler menu to the AdminBar so you can review the times that the plugin methods take to run.', 'jch-optimize' )
				],
				'order_plugin'            => [
					__( 'Order plugin', 'jch-optimize' ),
					__( 'If enabled, the plugin will ensure that the correct execution order of plugins to maintain compatibility with popular caching plugins are restored whenever a plugin is activated or deactivated.' )
				],
				'disable_logged_in_users' => [
					__( 'Disable logged in users', 'jch-optimize' ),
					__( 'When enabled, the plugin will be disabled for all users that are logged in', 'jch-optimize' )
				]
			],
			/*
			 * Exclude Menu Items
			 */
			'excludeMenuItemsSection' => [
				'menuexcludedurl' => [
					__( 'Exclude urls', 'jch-optimize' ),
					__( 'Enter a substring of the url you want to exclude here. Just type the string in the textbox then click the \'Add Item\' button for each url then save your settings', 'jch-optimize' )
				]
			],
			/**
			 * Combine CSS JS
			 */
			'combineCssJsSection'     => [
				'combine_files_enable' => [
					__( 'Enable', 'jch-optimize' ),
				],
				'pro_smart_combine'    => [
					__( 'Smart combine', 'jch-optimize' ),
					__( 'Will try to combine system and template files together respectively that are consistent across pages. <p>This produces several smaller combined files promoting browser caching across page loads; reduces chances of excessive cache generation and takes advantages of better files delivery offered by Http/2 servers', 'jch-optimize' )
				],
				'cache_lifetime'       => [
					__( 'Cache lifetime', 'jch-optimize' ),
					__( 'The lifetime of the cache files generated by the plugin. <p>If you\'re using a Page Cache plugin be sure to set this higher than the lifetime set in Page Cache. <p>If you\'re having issue with excess amount of cache being generated then lowering this setting will help.', 'jch-optimize' )
				],
				'html_minify_level'    => [
					__( 'HTML Minification level', 'jch-optimize' ),
					__( 'If \'Minify HTML\' is enabled, this will determine the level of minification. The incremental changes per level are as follows: <dl><dt>Basic - </dt><dd>Adjoining whitespaces outside of elements are reduced to one whitespace, HTML comments preserved;</dd><dt>Advanced - </dt><dd>Remove HTML comments, whitespace around block elements and undisplayed elements, Remove unnecessary whitespaces inside of elements and around their attributes;</dd> <dt>Ultra -</dt> <dd>Remove redundant attributes, for example, \'text/javascript\', and remove quotes from around selected attributes (HTML5)</dd> </dl>', 'jch-optimize' )
				],
				'htaccess'             => [
					__( 'Combined files delivery', 'jch-optimize' ),
					__( 'By default the combined files will be loaded as static css and javascript files. You would need to include directives in your .htaccess file to gzip these files. <p>You can use PHP files instead that will be gzipped if that option is set. PHP files can be loaded with a query attached with the information to find the combined files, or you can use url rewrite if it\'s available on the server so the files can be masked as static files. If your server prohibits the use of the Options +FollowSymLinks directive in .htaccess files use the respective option.', 'jch-optimize' )
				],
				'try_catch'            => [
					__( 'Use try-catch', 'jch-optimize' ),
					__( 'If you\'re seeing javascript errors in the console, you can try enabling this option to wrap each javascript file in a \'try-catch\' block to prevent the errors from one file affecting the combined file.', 'jch-optimize' )
				]
			],
			/**
			 *Combine Css Js Auto Settings
			 */
			'combineCssJsAutoSection' => [
				'gzip'                 => [
					__( 'GZip javascript and CSS', 'jch-optimize' ),
					__( 'The plugin will compress the combined JavaScript and CSS file respectively using gzip, if you\'ve selected one of the PHP options for \'Combined files delivery\'. This can decrease file size dramatically.', 'jch-optimize' )
				],
				'html_minify'          => [
					__( 'Minify HTML', 'jch-optimize' ),
					__( 'The plugin will remove all unnecessary whitespaces and comments from the HTML to reduce the total size of the web page.', 'jch-optimize' )
				],
				'includeAllExtensions' => [
					__( 'Include all plugins', 'jch-optimize' ),
					__( 'By default, all files from third party plugins and external domains are excluded. If enabled, they will be included.', 'jch-optimize' )
				],
				'phpAndExternal'       => [
					__( 'Include PHP and external files', 'jch-optimize' ),
					__( 'Javascript and css files with \'.php\' file extensions, and files from external domains will be included in the combined file. <p> This option requires that either cURL is installed on your server or your php option \'allow_url_fopen\' is enabled.' )
				]
			]
		];
	}

	public static function cssTab()
	{
		return [
			/**
			 * CSS Automatic Settings
			 */
			'cssAutoSettingsSection'     => [
				'css'            => [
					__( 'Combine CSS files', 'jch-optimize' ),
					__( 'This will combine all CSS files into one file and remove all the links to the individual files from the page, replacing them with a link generated by the plugin to the combined file.', 'jch-optimize' )
				],
				'css_minify'     => [
					__( 'Minify CSS', 'jch-optimize' ),
					__( 'The plugin will remove all unnecessary whitespaces and comments from the combined CSS file to reduce the total file size.', 'jch-optimize' )
				],
				'replaceImports' => [
					__( 'Replace @imports in CSS', 'jch-optimize' ),
					__( 'The plugin will replace <span class="notranslate">@import</span> at-rules with the contents of the files they are importing. This will be done recursively.', 'jch-optimize' )
				],
				'inlineStyle'    => [
					__( 'Include in-page &lt;style&gt;\'s' ),
					__( 'In-page CSS inside <span class="notranslate">&lt;style&gt;</span> tags will be included in the aggregated file in the order they appear on the page.', 'jch-optimize' )
				]
			],
			/**
			 * Exclude CSS Settings
			 */
			'excludeCssSection'          => [
				'excludeCSS'           => [
					__( 'Exclude CSS files', 'jch-optimize' ),
					__( 'Select the CSS files you want to exclude.', 'jch-optimize' )
				],
				'excludeCssComponents' => [
					__( 'Exclude CSS from these plugins', 'jch-optimize' ),
					__( 'The plugin will exclude all CSS files from the extensions you have selected.', 'jch-optimize' )
				],
				'excludeStyles'        => [
					__( 'Exclude individual in-page &lt;style&gt;\'s', 'jch-optimize' ),
					__( 'Select the internal <span class="notranslate">&lt;style&gt;</span> you want to exclude.', 'jch-optimize' )
				],
				'excludeAllStyles'     => [
					__( 'Exclude all in-page &lt;style&gt;\'s', 'jch-optimize' ),
					__( 'This is useful if you are generating an excess amount of cache files due to the file name of the combined CSS file keeps changing and you can\'t identify which STYLE declaration is responsible', 'jch-optimize' )
				]
			],
			/**
			 * Remove CSS
			 */
			'removeCssSection'           => [
				'remove_css' => [
					__( 'Remove CSS Files', 'jch-optimize' ),
					__( 'Select or add the files you want to prevent from loading on the site\'s pages.', 'jch-optimize' )
				]
			],
			/**
			 * Optimize Google Fonts
			 */
			'optimizeGoogleFontsSection' => [
				'pro_optimize_gfont_enable' => [
					__( 'Enable', 'jch-optimize' )
				]
			],
			/**
			 * Optimize CSS Delivery
			 */
			'optimizeCssDeliverySection' => [
				'optimizeCssDelivery_enable' => [
					__( 'Enable', 'jch-optimize' ),
				],
				'optimizeCssDelivery'        => [
					__( 'Number of elements', 'jch-optimize' ),
				],
				'pro_remove_unused_css'      => [
					__( 'Remove unused CSS', 'jch-optimize' ),
					__( 'When enabled, the plugin will \'lazy-load\' the combined CSS file only after the page is interacted with to prevent unnecessary processing of unused CSS before the page is loaded.', 'jch-optimize' )
				],
				'pro_dynamic_selectors'      => [
					__( 'CSS dynamic selectors', 'jch-optimize' ),
					__( 'In some cases when Remove Unused CSS is enabled, you may need to add the CSS for some dynamic elements to the critical CSS for them to load properly. Add any substring from the CSS declaration here to have them included.', 'jch-optimize' )
				]
			]
		];
	}

	/**
	 * Settings found on the Javascript Tab
	 *
	 * @return array
	 */

	public static function javascriptTab()
	{
		return [
			/**
			 * Javascript Automatic Settings
			 */
			'javascriptAutomaticSettingsSection' => [
				'javascript'       => [
					__( 'Combine javascript files', 'jch-optimize' ),
					__( 'This will combine all javascript files into one file and remove all the links to the individual files from the page, replacing them with a link generated by the plugin to the combined file.', 'jch-optimize' )
				],
				'js_minify'        => [
					__( 'Minify javascript', 'jch-optimize' ),
					__( 'The plugin will remove all unnecessary whitespaces and comments from the combined javascript file to reduce the total file size.', 'jch-optimize' )
				],
				'inlineScripts'    => [
					__( 'Include in-page &lt;script&gt; declarations', 'jch-optimize' ),
					__( 'In-page javascript inside <span class="notranslate">&lt;script&gt;</span> tags will be included in the combined file in the order they appear on the page.', 'jch-optimize' )
				],
				'bottom_js'        => [
					__( 'Position javascript files at bottom of page', 'jch-optimize' ),
					__( 'Place combined javascript file at bottom of the page just before the ending &lt;/body&gt; tag.<p> If some javascript files are excluded while preserving execution order, only the last combined javascript file will be placed at the bottom of the page.<p> If this is disabled, the plugin only combines files found in the &lt;head&gt; section of the HTML. This option extends the search to the &lt;body&gt; section.', 'jch-optimize' )
				],
				'loadAsynchronous' => [
					__( 'Defer or load javascript asychronously', 'jch-optimize' ),
					__( 'The \'asnyc\' attribute is added to the combined javascript file so it will be loaded asynchronously to avoid render blocking and speed up download of the web page. <p> If other files/scripts are excluded while preserving execution order, the \'defer\' attribute is instead used and is added to the last combined file(s) following an excluded file/script. <p>This option only works when the combined javascript file is placed at the bottom of the page. ' )
				]
			],
			/**
			 * Exclude Preserving Execution Order
			 */
			'excludePeoSection'                  => [
				'excludeJs_peo'           => [
					__( 'Exclude javascript files', 'jch-optimize' ),
					__( 'Select or add the files you want to exclude', 'jch-optimize' )
				],
				'excludeJsComponents_peo' => [
					__( 'Exclude javascript files from these plugins', 'jch-optimize' ),
					__( 'The plugin will exclude all javascript files from the plugins you have selected.', 'jch-optimize' )
				],
				'excludeScripts_peo'      => [
					__( 'Exclude individual in-page &lt;script&gt;\'s', 'jch-optimize' ),
					__( 'Select the internal <span class="notranslate">&lt;script&gt;</span>\'s you want to exclude.', 'jch-optimize' )
				],
				'excludeAllScripts'       => [
					__( 'Exclude all internal &lt;script&gt; declarations', 'jch-optimize' ),
					__( 'This is useful if you are generating an excess amount of cache files due to the file name of the combined javascript file keeps changing and you can\'t identify which SCRIPT declaration is responsible', 'jch-optimize' )

				]
			],
			/**
			 * Exclude Ignoring Excecution Order
			 */
			'excludeIeoSection'                  => [
				'excludeJs'           => [
					__( 'Exclude javascript files', 'jch-optimize' ),
					__( 'Select or add the javascript files you want to exclude.', 'jch-optimize' )
				],
				'excludeJsComponents' => [
					__( 'Exclude javascript files from these plugins', 'jch-optimize' ),
					__( 'The plugin will exclude all javascript files from the plugins you have selected.', 'jch-optimize' )
				],
				'excludeScripts'      => [
					__( 'Exclude individual in-page &lt;script&gt; declarations', 'jch-optimize' ),
					__( 'Select the internal <span class="notranslate">&lt;script&gt;</span> you want to exclude.', 'jch-optimize' )
				]
			],
			/**
			 * Don't Move Files To Bottom
			 */
			'dontMoveSection'                    => [
				'dontmoveJs'      => [
					__( 'Javascript files', 'jch-optimize' ),
					__( 'Don\'t move these javascript files (<i> that were excluded in the above sections</i> ) to the bottom of the page. These files will be left at their original position on the page.', 'jch-optimize' )
				],
				'dontmoveScripts' => [
					__( 'Inline scripts', 'jch-optimize' ),
					__( 'Enter any substring of an excluded script here to prevent this script being moved to the bottom. <p> Inline scripts in the &lt;script&gt; of the document containing the \'document.write\' method will NOT be moved by default.', 'jch-optimize' )
				]
			],
			/**
			 * Remove JS file
			 */
			'removeJsSection'                    => [
				'remove_js' => [
					__( 'Remove javascript files', 'jch-optimize' ),
					__( 'Select or add the javascript files you want to prevent from loading on the site\'s pages.', 'jch-optimize' )
				]
			]
		];
	}

	/**
	 * Settings on Page Cache
	 *
	 * @return array
	 */
	public static function pageCacheTab()
	{
		return [
			/**
			 * Page Cache
			 */
			'pageCacheSection' => [
				'cache_enable'        => [
					__( 'Enable', 'jch-optimize' ),
					__( 'Enable page caching', 'jch-optimize' )
				],
				'pro_cache_platform'  => [
					__( 'Platform specific', 'jch-optimize' ),
					__( 'Enable if HTML output on mobile differs from desktop.', 'jch-optimize' )
				],
				'page_cache_lifetime' => [
					__( 'Cache lifetime', 'jch-optimize' ),
					__( 'The period of time for which the page cache will be valid. Be sure to set this lower that the cache lifetime of combined files at all times.', 'jch-optimize' )
				],
				'cache_exclude'       => [
					__( 'Exclude urls', 'jch-optimize' ),
					__( 'Enter any part of a url to exclude that page from caching.', 'jch-optimize' )
				]
			]
		];
	}

	/**
	 * Settings on Media Tab
	 *
	 * @return array
	 */
	public static function mediaTab()
	{
		return [
			/**
			 * Add image Attributes
			 */
			'addImageAttributesSection' => [
				'img_attributes_enable' => [
					__( 'Enable', 'jch-optimize' )
				]
			],
			/**
			 * Sprite Generator
			 */
			'spriteGeneratorSection'    => [
				'csg_enable'         => [
					__( 'Enable', 'jch-optimize' )
				],
				'csg_direction'      => [
					__( 'Sprite build direction', 'jch-optimize' ),
					__( 'Determine in which direction the images must be placed in sprite.', 'jch-optimize' )
				],
				'csg_wrap_images'    => [
					__( 'Wrap images', 'jch-optimize' ),
					__( 'Will wrap images in sprite into another row or column if the length of the sprite becomes longer than <span class="notranslate">2000px</span>.', 'jch-optimize' ),
				],
				'csg_exclude_images' => [
					__( 'Exclude image from sprite', 'jch-optimize' ),
					__( 'You can exclude one or more of the images if they are displayed incorrectly.', 'jch-optimize' )
				],
				'csg_include_images' => [
					__( 'Include additional images in sprite', 'jch-optimize' ),
					__( 'You can include additional images in the sprite to the ones that were selected by default. Use this option cautiously, as these files are likely to display incorrectly.', 'jch-optimize' )
				]
			],
			/**
			 * Lazy-Load Images
			 */
			'lazyLoadSection'           => [
				'lazyload_enable'            => [
					__( 'Enable', 'jch-optimize' ),
				],
				'lazyload_autosize'          => [
					__( 'Autosize images', 'jch-optimize' ),
					__( 'This setting attempts to maintain aspect ratio of images being lazy-loaded to prevent blank spaces under the images after they\'re loaded or distortions in rendering.' )
				],
				'pro_lazyload_effects'       => [
					__( 'Enable effects', 'jch-optimize' ),
					__( 'Enable to use fade-in effects when images are scrolled into view.', 'jch-optimize' )
				],
				'pro_lazyload_iframe'        => [
					__( 'Lazy load iframes', 'jch-optimize' ),
					__( 'If enabled will also lazyload &lt;iframe&gt; elements.', 'jch-optimize' ),
				],
				'pro_lazyload_bgimages'      => [
					__( 'Background images', 'jch-optimize' ),
					__( 'Will lazyload background images defined in \'style\' attributes on HTML elements.', 'jch-optimize' )
				],
				'pro_lazyload_audiovideo'    => [
					__( 'Audio/Video', 'jch-optimize' ),
					__( 'Will lazyload &lt;audio&gt; and &lt;video&gt; elements that are below the fold.' )
				],
				'excludeLazyLoad'            => [
					__( 'Exclude these images', 'jch-optimize' ),
					__( 'Select or manually add the urls of the images you want to exclude from lazy load.', 'jch-optimize' )
				],
				'pro_excludeLazyLoadFolders' => [
					__( 'Exclude these folders', 'jch-optimize' ),
					__( 'Exclude all the images in the selected folders.', 'jch-optimize' )
				],
				'pro_excludeLazyLoadClass'   => [
					__( 'Exclude these classes', 'jch-optimize' ),
					__( 'Exclude all images that has these classes declared on the <span class="notranslate">&lt;img&gt;</span> element', 'jch-optimize' )
				]
			]
		];
	}

	/**
	 * Settings on Http/2 tab
	 *
	 * @return array
	 */
	public static function http2PushTab()
	{
		return [
			/**
			 * Http2 Push
			 */
			'http2PushSection' => [
				'http2_push_enable'          => [
					__( 'Enable', 'jch-optimize' ),
				],
				'pro_http2_exclude_deferred' => [
					__( 'Exclude deferred files', 'jch-optimize' ),
					__( 'Will exclude javascript files that are deferred or loaded asynchronously, deferred CSS file in Optimize CSS Delivery feature, and images that are lazy-loaded. This can help reduce bandwidth and speed up first paint rendering.', 'jch-optimize' )
				],
				'pro_http2_push_cdn'         => [
					__( 'Push CDN files', 'jch-optimize' ),
					__( 'Files loaded over your CDN domains will also be included in the Link headers', 'jch-optimize' )
				],
				'pro_http2_file_types'       => [
					__( 'File types', 'jch-optimize' ),
					__( 'Select the file types you want pushed by http/2', 'jch-optimize' )
				],
				'pro_http2_include'          => [
					__( 'Include files', 'jch-optimize' ),
					__( 'Sometimes some files are dynamically loaded so you can add these files here. Be sure any file added here are loaded on all pages and that you include the full file path including any queries etc. Only the following file extensions are supported: .js, .css, .webp, .gif, .png, .jpg, .woff, .woff2', 'jch-optimize' )
				],
				'pro_http2_exclude'          => [
					__( 'Exclude files', 'jch-optimize' ),
					__( 'If you see any warnings in the browser console that the preloaded files weren\'t used within a few seconds you can exclude these files here', 'jch-optimize' )
				]
			],
		];
	}

	/**
	 * Settings on CDN tab
	 *
	 * @return array
	 */
	public static function cdnTab()
	{
		return [
			/**
			 * CDN
			 */
			'cdnSection' => [
				'cookielessdomain_enable' => [
					__( 'Enable', 'jch-optimize' ),
				],
				'pro_cdn_preconnect'      => [
					__( 'Preconnect CDNs', 'jch-optimize' ),
					__( 'Add preconnect resource hints in the HTML for all your CDN domains to establish early connections and speed up loading of resources from your CDN.', 'jch-optimize' )
				],
				'cdn_scheme'              => [
					__( 'CDN scheme', 'jch-optimize' ),
					__( 'Select the scheme that you want prepended to the CDN/Cookieless domain', 'jch-optimize' )
				],
				'cookielessdomain'        => [
					__( 'Domain 1', 'jch-optimize' ),
					__( 'Enter value for Domain #1 here', 'jch-optimize' )
				],
				'staticfiles'             => [
					__( 'Static files 1', 'jch-optimize' ),
					__( 'Select the static file types that you want to be loaded over Domain #1', 'jch-optimize' )
				],
				'pro_customcdnextensions' => [
					__( 'Custom extensions 1', 'jch-optimize' ),
					__( 'To add custom extensions of file types to be loaded over CDN on Domain 1, type the extension in the textbox and press the \'Add item\' button', 'jch-optimize' )
				],
				'pro_cookielessdomain_2'  => [
					__( 'Domain 2', 'jch-optimize' ),
					__( 'Enter value for Domain #2 here', 'jch-optimize' )
				],
				'pro_staticfiles_2'       => [
					__( 'Static files 2', 'jch-optimize' ),
					__( 'Select the static file types that you want to be loaded over Domain #2', 'jch-optimize' )
				],
				'pro_cookielessdomain_3'  => [
					__( 'Domain 3', 'jch-optimize' ),
					__( 'Enter value for Domain #3 here', 'jch-optimize' )
				],
				'pro_staticfiles_3'       => [
					__( 'Static files 3', 'jch-optimize' ),
					__( 'Select the static file types that you want to be loaded over Domain #3', 'jch-optimize' )
				]
			]
		];
	}

	/**
	 * Settings on Optimize Image tab
	 *
	 * @return array
	 */
	public static function optimizeImage()
	{
		return [
			/**
			 * Global Section
			 */
			'globalSection'    => [
				'ignore_optimized'     => [
					__( 'Ignore optimized images', 'jch-optimize' ),
					__( 'Will not attempt to optimize any images in subfolders that have already been marked as optimized.', 'jch-optimize' )
				],
				'pro_next_gen_images'  => [
					__( 'Next-Gen images', 'jch-optimize' ),
					__( 'When enabled the plugin will convert the images that are optimized to webp format and load these instead.', 'jch-optimize' )
				],
				'pro_web_old_browsers' => [
					__( 'Support old browsers', 'jch-optimize' ),
					__( 'Plugin will wrap WEBP image in a &lt;picture/&gt; element along with original image so browsers without WEBP support can fall back to the original image.', 'jch-optimize' )
				],
				'lossy'                => [
					__( 'Optimization level', 'jch-optimize' ),
					__( 'Levels are either Lossy or Lossless, the default is Lossy. With Lossy optimization images will be more optimized and smaller but may result in a small reduction of quality, most times invisible to the untrained eye. If you don\'t want that then you can choose Lossless instead. The images will not be as optimized but there will be no loss of quality.', 'jch-optimize' )
				],
				'save_metadata'        => [
					__( 'Save metadata', 'jch-optimize' ),
					__( 'The Optimize Image API will remove all metadata from images while optimizing including any copyrights or Exif textual content not part of the actual image to produce the smallest possible file size. If you wish to retain this information at a small loss in optimization then you should enable this option.', 'jch-optimize' )
				]
			],
			/**
			 * Automatically Optimize Section
			 */
			'autoApiSection'   => [
				'pro_api_resize_mode' => [
					__( 'Auto resize images', 'jch-optimize' ),
					__( 'Images will be resized automatically to the dimensions as shown on screen. If you use an application that shows the images larger when you hover over them be sure to use different images for the large ones. For this to work the url to your site must be available publicly.', 'jch-optimize' )
				]
			],
			/**
			 * Manually optimize Section
			 */
			'manualApiSection' => [
				'recursive' => [
					__( 'Recurse in subfolders', 'jch-optimize' ),
					__( 'Will optimize all images in selected folders and recurse into subfolders infinitely. If disabled, only the images in the selected folders will be optimized. Subfolders will be ignored.', 'jch-optimize' )
				]
			]
		];
	}

	/**
	 * Settings on Miscellanous tab
	 *
	 * @return array
	 */
	public static function miscellaneous()
	{
		return [
			/**
			 * Reduce Dom Section
			 */
			'reduceDomSection' => [
				'pro_reduce_dom'    => [
					__( 'Enable', 'jch-optimize' )
				],
				'pro_html_sections' => [
					__( 'HTML sections', 'jch-optimize' ),
					__( 'Select which HTML sections you would like to load asynchronously.' )
				]
			]
		];
	}
}